package com.beem.project.beem.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jivesoftware.smack.util.StringUtils;

import android.app.ExpandableListActivity;
import android.content.ComponentName;
import android.content.Intent;
import android.content.ServiceConnection;
import android.database.DataSetObserver;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.View.OnClickListener;
import android.view.View.OnLongClickListener;
import android.widget.ExpandableListAdapter;
import android.widget.ImageView;
import android.widget.TextView;

import com.beem.project.beem.BeemService;
import com.beem.project.beem.R;
import com.beem.project.beem.service.Contact;
import com.beem.project.beem.service.PresenceAdapter;
import com.beem.project.beem.service.aidl.IBeemRosterListener;
import com.beem.project.beem.service.aidl.IRoster;
import com.beem.project.beem.service.aidl.IXmppFacade;
import com.beem.project.beem.utils.Status;

public class ContactList extends ExpandableListActivity {

    private static final String TAG = "CONTACTLIST_ACT";
    public static final String DEFAULT_GROUP = "Default";
    private MyExpandableListAdapter mAdapter;
    private IRoster mRoster;
    private Map<String, List<Contact>> groupMap;
    private List<String> groupName;
    private List<Contact> mListContact;
    private Handler mHandler;
    private IXmppFacade xmppFacade = null;
    private final ServiceConnection mServConn = new BeemServiceConnection();
    private int REQUEST_CODE = 1;

    /**
     * Callback for menu creation.
     * @param menu the menu created
     * @return true on success, false otherwise
     */
    @Override
    public final boolean onCreateOptionsMenu(Menu menu) {
	super.onCreateOptionsMenu(menu);
	MenuInflater inflater = getMenuInflater();
	inflater.inflate(R.menu.contact_list, menu);
	return true;
    }

    /**
     * Callback for menu item selected.
     * @param item the item selected
     * @return true on success, false otherwise
     */
    @Override
    public final boolean onOptionsItemSelected(MenuItem item) {
	switch (item.getItemId()) {
	    case R.id.contact_list_menu_settings:
		startActivityForResult(new Intent(this, EditSettings.class), REQUEST_CODE);
		return true;
	    case R.id.contact_list_menu_add_contact:
		startActivity(new Intent(ContactList.this, AddContact.class));
		return true;
	    default:
		return false;
	}
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
	super.onActivityResult(requestCode, resultCode, data);
	if (requestCode == REQUEST_CODE) {
	    if (resultCode == 69) {
		finish();
		stopService(new Intent(this, BeemService.class));
		startActivity(new Intent(this, Login.class));
	    }
	}
    }

    @Override
    protected void onCreate(Bundle saveBundle) {
	super.onCreate(saveBundle);
	mHandler = new Handler();
	groupMap = new HashMap<String, List<Contact>>();
	groupName = new ArrayList<String>();
    }

    @Override
    protected void onStart() {
	super.onStart();
	bindService(new Intent(this, BeemService.class), mServConn, BIND_AUTO_CREATE);
    }

    @Override
    protected void onStop() {
	super.onStop();
	unbindService(mServConn);
	groupName.clear();
	groupMap.clear();
    }

    class ComparatorContactListByName<T> implements Comparator<T> {
	@Override
	public int compare(T c1, T c2) {
	    return ((Contact) c1).getName().compareToIgnoreCase(((Contact) c2).getName());
	}
    }

    class ComparatorContactListByStatusAndName<T> implements Comparator<T> {
	@Override
	public int compare(T c1, T c2) {
	    if (((Contact) c1).getStatus() < ((Contact) c2).getStatus()) {
		return 1;
	    } else if (((Contact) c1).getStatus() > ((Contact) c2).getStatus()) {
		return -1;
	    } else
		return ((Contact) c1).getName().compareToIgnoreCase(((Contact) c2).getName());
	}
    }

    private void buildContactList(List<Contact> listContact) {
	mListContact = listContact;
	Collections.sort(mListContact, new ComparatorContactListByStatusAndName<Contact>());
	Collections.sort(groupName);
	for (Contact contact : mListContact) {
	    for (String group : contact.getGroups()) {
		addGroup(group);
		addContactInGroup(contact, group);
	    }
	    if (contact.getGroups().isEmpty()) {
		addGroup(DEFAULT_GROUP);
		addContactInGroup(contact, DEFAULT_GROUP);
	    }
	}
	Collections.sort(groupName);
	mAdapter = new MyExpandableListAdapter();
	setListAdapter(mAdapter);
    }

    protected void addGroup(String group) {
	if (!groupMap.containsKey(group)) {
	    groupMap.put(group, new ArrayList<Contact>());
	    groupName.add(group);
	}
    }

    protected void addContactInGroup(Contact c, String group) {
	boolean found = false;
	for (Contact tmpContact : groupMap.get(group)) {
	    if (c.getJID().equals(tmpContact.getJID())) {
		found = true;
		break;
	    }
	}
	if (!found)
	    groupMap.get(group).add(c);
    }

    protected void delContactInGroup(Contact c, String group) {
	groupMap.get(group).remove(c);
	c.delGroup(group);
	if (groupMap.get(group).isEmpty()) {
	    groupMap.remove(group);
	    groupName.remove(group);
	}
    }

    private class BeemRosterListener extends IBeemRosterListener.Stub {

	@Override
	public void onEntriesAdded(List<String> addresses) throws RemoteException {
	    for (String str : addresses) {
		Contact curContact = mRoster.getContact(str);
		for (String group : curContact.getGroups()) {
		    addGroup(group);
		    addContactInGroup(curContact, group);
		}
	    }
	    mHandler.post(new RunnableChange());
	}

	@Override
	public void onEntriesDeleted(List<String> addresses) throws RemoteException {
	    for (String user : addresses) {
		List<Contact> tmpListContact = groupMap.get(DEFAULT_GROUP);
		for (Contact contact : tmpListContact) {
		    if (contact.getJID().equals(user)) {
			delContactInGroup(contact, DEFAULT_GROUP);
			break;
		    }
		}
	    }
	    mHandler.post(new RunnableChange());
	}

	@Override
	public void onEntriesUpdated(List<String> addresses) throws RemoteException {
	    for (String str : addresses) {
		Contact curContact = mRoster.getContact(str);
		for (String group : curContact.getGroups()) {
		    addGroup(group);
		    addContactInGroup(curContact, group);
		}
	    }
	    mHandler.post(new RunnableChange());
	}

	@Override
	public void onPresenceChanged(PresenceAdapter presence) throws RemoteException {
	    for (Contact curContact : mListContact) {
		if (curContact.getJID().equals(StringUtils.parseBareAddress(presence.getFrom()))) {
		    curContact.setStatus(mRoster.getPresence(StringUtils.parseBareAddress(presence.getFrom())));
		    mHandler.post(new RunnableChange());
		    return;
		}
	    }
	}

	private class RunnableChange implements Runnable {
	    @Override
	    public void run() {
		mAdapter.changed();
	    }
	}

	@Override
	public void onEntryDeleteFromGroup(String group, String jid) throws RemoteException {
	    for (Contact contact : mListContact) {
		if (jid.equals(contact.getJID())) {
		    delContactInGroup(contact, group);
		    if (contact.getGroups().size() == 0) {
			addGroup(DEFAULT_GROUP);
			addContactInGroup(contact, DEFAULT_GROUP);
		    }
		    break;
		}
	    }
	    mHandler.post(new RunnableChange());
	}
    }

    private class MyExpandableListAdapter implements ExpandableListAdapter {

	class MyOnClickListener implements OnClickListener {

	    private final Contact mContact;

	    public MyOnClickListener(Contact contact) {
		mContact = contact;
	    }

	    @Override
	    public void onClick(View v) {
		Intent i = new Intent(ContactList.this, SendIM.class);
		i.setData(mContact.toUri());
		startActivity(i);
	    }

	}

	class MyOnLongClickListener implements OnLongClickListener {

	    private final Contact mContact;
	    private final String mGroup;

	    public MyOnLongClickListener(Contact contact, String group) {
		mContact = contact;
		mGroup = group;
	    }

	    @Override
	    public boolean onLongClick(View v) {
		createDialog(mContact, mGroup);
		return true;
	    }
	}

	private final List<DataSetObserver> observers;

	public MyExpandableListAdapter() {
	    observers = new ArrayList<DataSetObserver>();
	}

	@Override
	public boolean areAllItemsEnabled() {
	    return true;
	}

	private void bindView(View view, Contact curContact) {

	    if (curContact != null) {
		ImageView imgV = (ImageView) view.findViewById(R.id.contactliststatus);
		TextView v = (TextView) view.findViewById(R.id.contactlistpseudo);
		Drawable imageDrawable = null;
		switch (curContact.getStatus()) {
		    case Status.CONTACT_STATUS_AVAILABLE:
			imageDrawable = getResources().getDrawable(R.drawable.online);
			v.setTextColor(getResources().getColor(R.color.white));
			break;
		    case Status.CONTACT_STATUS_AVAILABLE_FOR_CHAT:
			imageDrawable = getResources().getDrawable(R.drawable.chat);
			break;
		    case Status.CONTACT_STATUS_AWAY:
			imageDrawable = getResources().getDrawable(R.drawable.away);
			break;
		    case Status.CONTACT_STATUS_BUSY:
			imageDrawable = getResources().getDrawable(R.drawable.dnd);
			break;
		    case Status.CONTACT_STATUS_DISCONNECT:
			imageDrawable = getResources().getDrawable(R.drawable.offline);
			break;
		    case Status.CONTACT_STATUS_UNAVAILABLE:
			imageDrawable = getResources().getDrawable(R.drawable.requested);
			break;
		    default:
			imageDrawable = getResources().getDrawable(R.drawable.error);
			break;
		}
		imgV.setImageDrawable(imageDrawable);

		String mContactName = curContact.getName();
		if ("".equals(mContactName)) {
		    mContactName = curContact.getJID();
		    mContactName = StringUtils.parseName(mContactName);
		    if ("".equals(mContactName))
			mContactName = curContact.getJID();
		}
		v.setText(mContactName);

		v = (TextView) view.findViewById(R.id.contactlistmsgperso);
		if (v != null) {
		    v.setText(curContact.getMsgState());
		}

		/*
		 * TODO: Rajouter l'avatar du contact getAvatar() dans la classe imgV = (ImageView)
		 * view.findViewById(R.id.contactlistavatar); if (imgV != null) { imageDrawable =
		 * getResources().getDrawable(R.drawable.avatar); imgV.setImageDrawable(imageDrawable); }
		 */
	    }
	}

	public void changed() {
	    Collections.sort(groupName);
	    for (String name : groupName) {
		Collections.sort(groupMap.get(name), new ComparatorContactListByStatusAndName<Contact>());
	    }
	    for (DataSetObserver obs : observers) {
		obs.onChanged();
	    }
	}

	void createDialog(Contact contact, String group) {
	    ContactDialog dialogContact = new ContactDialog(ContactList.this, contact, group);
	    dialogContact.setOwnerActivity(ContactList.this);
	    dialogContact.show();
	}

	@Override
	public Object getChild(int groupPosition, int childPosition) {
	    try {
		return groupMap.get(groupName.get(groupPosition)).get(childPosition);
	    } catch (NullPointerException e) {
		Log.e(TAG, "Child not found", e);
		return null;
	    }
	}

	@Override
	public long getChildId(int groupPosition, int childPosition) {
	    try {
		groupMap.get(groupName.get(groupPosition)).get(childPosition);
	    } catch (NullPointerException e) {
		Log.e(TAG, "Child not found", e);
		return 0;
	    }
	    return childPosition;
	}

	@Override
	public int getChildrenCount(int groupPosition) {
	    try {
		return groupMap.get(groupName.get(groupPosition)).size();
	    } catch (NullPointerException e) {
		Log.e(TAG, "Child not found", e);
		return 0;
	    }
	}

	@Override
	public View getChildView(int groupPosition, int childPosition, boolean isLastChild, View convertView,
	    ViewGroup parent) {
	    View v;
	    if (convertView == null) {
		v = LayoutInflater.from(ContactList.this).inflate(R.layout.contactlistcontact, null);
	    } else {
		v = convertView;
	    }
	    Contact contact = groupMap.get(groupName.get(groupPosition)).get(childPosition);
	    bindView(v, contact);

	    v.setOnLongClickListener(new MyOnLongClickListener(contact, groupName.get(groupPosition)));
	    v.setOnClickListener(new MyOnClickListener(contact));
	    return v;
	}

	@Override
	public long getCombinedChildId(long groupId, long childId) {
	    return 1000 * groupId + childId;
	}

	@Override
	public long getCombinedGroupId(long groupId) {
	    return 1000 * groupId;
	}

	@Override
	public Object getGroup(int groupPosition) {
	    try {
		return groupMap.get(groupName.get(groupPosition));
	    } catch (NullPointerException e) {
		Log.e(TAG, "Group not found", e);
		return null;
	    }
	}

	@Override
	public int getGroupCount() {
	    return groupMap.size();
	}

	@Override
	public long getGroupId(int groupPosition) {
	    return groupPosition;
	}

	@Override
	public View getGroupView(int groupPosition, boolean isExpanded, View convertView, ViewGroup parent) {
	    if (convertView == null) {
		convertView = LayoutInflater.from(ContactList.this).inflate(R.layout.contactlistgroup, null);
	    }
	    TextView groupTextView = (TextView) convertView.findViewById(R.id.textgroup);
	    groupTextView.setText(groupName.get(groupPosition));
	    return convertView;
	}

	@Override
	public boolean hasStableIds() {
	    return false;
	}

	@Override
	public boolean isChildSelectable(int groupPosition, int childPosition) {
	    return true;
	}

	@Override
	public boolean isEmpty() {
	    return groupMap.isEmpty();
	}

	@Override
	public void onGroupCollapsed(int groupPosition) {
	}

	@Override
	public void onGroupExpanded(int groupPosition) {
	}

	@Override
	public void registerDataSetObserver(DataSetObserver observer) {
	    observers.add(observer);
	}

	@Override
	public void unregisterDataSetObserver(DataSetObserver observer) {
	    observers.remove(observer);
	}
    }

    private class BeemServiceConnection implements ServiceConnection {
	BeemRosterListener mBeemRosterListener = new BeemRosterListener();

	@Override
	public void onServiceConnected(ComponentName name, IBinder service) {
	    xmppFacade = IXmppFacade.Stub.asInterface(service);
	    try {
		mRoster = xmppFacade.getRoster();
		if (mRoster != null) {
		    mRoster.addRosterListener(mBeemRosterListener);
		    buildContactList(mRoster.getContactList());
		}
	    } catch (RemoteException e) {
		e.printStackTrace();
	    }
	}

	@Override
	public void onServiceDisconnected(ComponentName name) {
	    try {
		mRoster.removeRosterListener(mBeemRosterListener);
	    } catch (RemoteException e) {
		e.printStackTrace();
	    }
	    xmppFacade = null;
	    mRoster = null;
	}
    }
}
