/*
    BEEM is a videoconference application on the Android Platform.

    Copyright (C) 2009 by Frederic-Charles Barthelery,
                          Jean-Manuel Da Silva,
                          Nikita Kozlov,
                          Philippe Lago,
                          Jean Baptiste Vergely,
                          Vincent Veronis.

    This file is part of BEEM.

    BEEM is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    BEEM is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with BEEM.  If not, see <http://www.gnu.org/licenses/>.

    Please send bug reports with examples or suggestions to
    contact@beem-project.com or http://dev.beem-project.com/

    Epitech, hereby disclaims all copyright interest in the program "Beem"
    written by Frederic-Charles Barthelery,
               Jean-Manuel Da Silva,
               Nikita Kozlov,
               Philippe Lago,
               Jean Baptiste Vergely,
               Vincent Veronis.

    Nicolas Sadirac, November 26, 2009
    President of Epitech.

    Flavien Astraud, November 26, 2009
    Head of the EIP Laboratory.

 */
package com.beem.project.beem.jingle;

import java.util.ArrayList;
import java.util.List;

import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.XMPPException;
import org.jivesoftware.smackx.jingle.JingleManager;
import org.jivesoftware.smackx.jingle.JingleSession;
import org.jivesoftware.smackx.jingle.JingleSessionRequest;
import org.jivesoftware.smackx.jingle.listeners.JingleSessionListener;
import org.jivesoftware.smackx.jingle.listeners.JingleSessionRequestListener;
import org.jivesoftware.smackx.jingle.media.JingleMediaManager;
import org.jivesoftware.smackx.jingle.media.PayloadType;
import org.jivesoftware.smackx.jingle.nat.BasicTransportManager;
import org.jivesoftware.smackx.jingle.nat.TransportCandidate;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.RemoteCallbackList;
import android.os.RemoteException;
import android.util.Log;

import com.beem.project.beem.service.aidl.IBeemJingleListener;
import com.beem.project.beem.service.aidl.IJingle;
import com.beem.project.beem.ui.Call;

/**
 * Beem Jingle Service, manage jingle call.
 * @author nikita
 */
public class JingleService extends IJingle.Stub {
    private static final String TAG = "JingleService";
    private JingleManager mJingleManager;
    private final List<JingleMediaManager> mMediaManagers;
    private final RemoteCallbackList<IBeemJingleListener> mRemoteJingleListeners = new RemoteCallbackList<IBeemJingleListener>();
    private JingleSession mIn;

    private JingleSession mOut;
    private JingleSessionRequest mRequest;
    private Context mContext;
    private boolean isCaller;
    private RTPAudioSession mAudioSession;

    /**
     * JingleService constructor.
     * @param xmppConnection a valid XMPPConnection
     */
    public JingleService(final Context ctx) {
	BasicTransportManager bt = new BasicTransportManager();
	mMediaManagers = new ArrayList<JingleMediaManager>();
	mMediaManagers.add(new MicrophoneRTPManager(bt, ctx));
	mContext = ctx;
    }

    /**
     * finish to construct the instance.
     * @param conn the xmppConnection used with constructor
     */
    public void initWhenConntected(XMPPConnection conn) {
	mJingleManager = new JingleManager(conn, mMediaManagers);
	mJingleManager.addJingleSessionRequestListener(new BeemJingleSessionRequestListener());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addJingleListener(IBeemJingleListener listen) throws RemoteException {
	if (listen != null)
	    mRemoteJingleListeners.register(listen);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeJingleListener(IBeemJingleListener listen) throws RemoteException {
	if (listen != null)
	    mRemoteJingleListeners.unregister(listen);
    }

    /**
     * begin a jingle call.
     * @param receiver the call receiver
     */
    @Override
    public void call(final String receiver) throws RemoteException {
    	Log.d(TAG, "Place Call");
	try {
	    mOut = mJingleManager.createOutgoingJingleSession(receiver);
	    mOut.addListener(new BeemJingleSessionListener());
	    mOut.startOutgoing();
	    isCaller = true;
	} catch (XMPPException e) {
	    e.printStackTrace();
	}
    }


    @Override
    public void acceptCall() throws RemoteException {
    	Log.d(TAG, "Accept Call");
	try {
	    mRequest.accept();
	    mIn.start();
	} catch (XMPPException e) {
	    e.printStackTrace();
	}
	isCaller = false;
    }
    
    @Override
    public void setSpeakerMode(int mode) throws RemoteException {
	mAudioSession.setSpeakerMode(mode);
    }

    /**
     * close a jingle call.
     */
    @Override
    public void closeCall() throws RemoteException {
	mAudioSession = null;
	if (isCaller) {
	    try {
		mOut.terminate("Cancelled");
		mOut.close();
	    } catch (XMPPException e) {
		e.printStackTrace();
	    }
	    mOut = null;
	} else {
	    try {
		mRequest.reject();
		mIn.terminate();
		mIn.close();		    
	    } catch (XMPPException e) {
		e.printStackTrace();
	    }
	    mIn = null;
	}
    }

    /**
     * Listen on session events.
     * @author nikita
     */
    private class BeemJingleSessionListener implements JingleSessionListener {

	/**
	 * constructor.
	 */
	public BeemJingleSessionListener() {
	    super();
	}

	@Override
	public void sessionClosed(String reason, JingleSession jingleSession) {
	    System.out.println("Session " + jingleSession.getResponder() + " closed because " + reason);

	    final int n = mRemoteJingleListeners.beginBroadcast();
	    for (int i = 0; i < n; i++) {
		IBeemJingleListener listener = mRemoteJingleListeners.getBroadcastItem(i);
		try {
		    listener.sessionClosed(reason);
		} catch (RemoteException e) {
		    e.printStackTrace();
		}
	    }
	    mRemoteJingleListeners.finishBroadcast();
	}

	@Override
	public void sessionClosedOnError(XMPPException e, JingleSession jingleSession) {
	    System.out.println("Session " + jingleSession.getResponder() + " closed");

	    final int n = mRemoteJingleListeners.beginBroadcast();
	    for (int i = 0; i < n; i++) {
		IBeemJingleListener listener = mRemoteJingleListeners.getBroadcastItem(i);
		try {
		    listener.sessionClosedOnError(e.getMessage());
		} catch (RemoteException err) {
		    err.printStackTrace();
		}
	    }
	    mRemoteJingleListeners.finishBroadcast();
	}

	@Override
	public void sessionDeclined(String reason, JingleSession jingleSession) {
	    Log.d(TAG, "Session " + jingleSession.getResponder() + " declined because " + reason);

	    final int n = mRemoteJingleListeners.beginBroadcast();
	    for (int i = 0; i < n; i++) {
		IBeemJingleListener listener = mRemoteJingleListeners.getBroadcastItem(i);
		try {
		    listener.sessionDeclined(reason);
		} catch (RemoteException e) {
		    e.printStackTrace();
		}
	    }
	    mRemoteJingleListeners.finishBroadcast();

	}

	@Override
	public void sessionEstablished(PayloadType pt, TransportCandidate remoteCandidate,
	    TransportCandidate localCandidate, JingleSession jingleSession) {
	    Log.d(TAG, "Session " + jingleSession.getResponder() + " established");
	    mAudioSession = (RTPAudioSession) jingleSession.getSession().getMediaSession(MicrophoneRTPManager.MEDIA_NAME);
	    final int n = mRemoteJingleListeners.beginBroadcast();
	    for (int i = 0; i < n; i++) {
		IBeemJingleListener listener = mRemoteJingleListeners.getBroadcastItem(i);
		try {
		    listener.sessionEstablished();
		} catch (RemoteException e) {
		    e.printStackTrace();
		}
	    }
	    mRemoteJingleListeners.finishBroadcast();
	}

	@Override
	public void sessionMediaReceived(JingleSession jingleSession, String participant) {
	    Log.d(TAG, "Session Media received from " + participant);
	}

	@Override
	public void sessionRedirected(String redirection, JingleSession jingleSession) {
	}
    }


    /**
     * Listen for a Jingle session request.
     * @author nikita
     */
    private class BeemJingleSessionRequestListener implements JingleSessionRequestListener {

	/**
	 * Constructor.
	 */
	public BeemJingleSessionRequestListener() {
	    super();
	}

	@Override
	public void sessionRequested(JingleSessionRequest request) {
	    mRequest = request;
	    try {
		mIn = mJingleManager.createIncomingJingleSession(mRequest);
		mIn.addListener(new BeemJingleSessionListener());
		mIn.startIncoming();
	    } catch (XMPPException e) {
		e.printStackTrace();
	    }
	    System.out.println("Jingle Session request from " + request.getFrom());
	    isCaller = false;
	    Intent intent = new Intent(mContext, Call.class);
	    intent.setData(Uri.parse("jingle:"+request.getFrom()));
	    intent.putExtra("isCaller", false);
	    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
	    mContext.startActivity(intent);
	}
    }

}
