/*
    BEEM is a videoconference application on the Android Platform.

    Copyright (C) 2009 by Frederic-Charles Barthelery,
                          Jean-Manuel Da Silva,
                          Nikita Kozlov,
                          Philippe Lago,
                          Jean Baptiste Vergely,
                          Vincent Veronis.

    This file is part of BEEM.

    BEEM is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    BEEM is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with BEEM.  If not, see <http://www.gnu.org/licenses/>.

    Please send bug reports with examples or suggestions to
    contact@beem-project.com or http://dev.beem-project.com/

    Epitech, hereby disclaims all copyright interest in the program "Beem"
    written by Frederic-Charles Barthelery,
               Jean-Manuel Da Silva,
               Nikita Kozlov,
               Philippe Lago,
               Jean Baptiste Vergely,
               Vincent Veronis.

    Nicolas Sadirac, November 26, 2009
    President of Epitech.

    Flavien Astraud, November 26, 2009
    Head of the EIP Laboratory.

*/
package com.beem.project.beem;

import org.jivesoftware.smack.ConnectionConfiguration;
import org.jivesoftware.smack.Roster;
import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.ConnectionConfiguration.SecurityMode;
import org.jivesoftware.smack.Roster.SubscriptionMode;
import org.jivesoftware.smack.provider.PrivacyProvider;
import org.jivesoftware.smack.provider.ProviderManager;
import org.jivesoftware.smack.proxy.ProxyInfo;
import org.jivesoftware.smack.proxy.ProxyInfo.ProxyType;
import org.jivesoftware.smack.util.StringUtils;
import org.jivesoftware.smackx.GroupChatInvitation;
import org.jivesoftware.smackx.PrivateDataManager;
import org.jivesoftware.smackx.packet.ChatStateExtension;
import org.jivesoftware.smackx.packet.LastActivity;
import org.jivesoftware.smackx.packet.OfflineMessageInfo;
import org.jivesoftware.smackx.packet.OfflineMessageRequest;
import org.jivesoftware.smackx.packet.SharedGroupsInfo;
import org.jivesoftware.smackx.provider.AdHocCommandDataProvider;
import org.jivesoftware.smackx.provider.BytestreamsProvider;
import org.jivesoftware.smackx.provider.DataFormProvider;
import org.jivesoftware.smackx.provider.DelayInformationProvider;
import org.jivesoftware.smackx.provider.DiscoverInfoProvider;
import org.jivesoftware.smackx.provider.DiscoverItemsProvider;
import org.jivesoftware.smackx.provider.IBBProviders;
import org.jivesoftware.smackx.provider.MUCAdminProvider;
import org.jivesoftware.smackx.provider.MUCOwnerProvider;
import org.jivesoftware.smackx.provider.MUCUserProvider;
import org.jivesoftware.smackx.provider.MessageEventProvider;
import org.jivesoftware.smackx.provider.MultipleAddressesProvider;
import org.jivesoftware.smackx.provider.RosterExchangeProvider;
import org.jivesoftware.smackx.provider.StreamInitiationProvider;
import org.jivesoftware.smackx.provider.VCardProvider;
import org.jivesoftware.smackx.provider.XHTMLExtensionProvider;
import org.jivesoftware.smackx.search.UserSearch;

import android.app.Notification;
import android.app.NotificationManager;
import android.app.Service;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.os.IBinder;
import android.os.RemoteException;
import android.preference.PreferenceManager;
import android.util.Log;

import com.beem.project.beem.jingle.JingleService;
import com.beem.project.beem.service.XmppConnectionAdapter;
import com.beem.project.beem.service.XmppFacade;
import com.beem.project.beem.service.aidl.IXmppFacade;

// TODO: Auto-generated Javadoc
/**
 * This class is for the Beem service. The connection to the xmpp server will be made asynchronously when the service
 * will start.
 * @author darisk
 */
public class BeemService extends Service {

    /** The id to use for status notification. */
    public static final int NOTIFICATION_STATUS_ID = 100;

    private static final String TAG = "BeemService";
    private static final int DEFAULT_XMPP_PORT = 5222;
    private static final String COMMAND_NAMESPACE = "http://jabber.org/protocol/commands";

    private NotificationManager mNotificationManager;
    private XmppConnectionAdapter mConnection;
    private JingleService mJingle;
    private SharedPreferences mSettings;
    private String mLogin;
    private String mPassword;
    private String mHost;
    private String mService;
    private int mPort;
    private ConnectionConfiguration mConnectionConfiguration;
    private ProxyInfo mProxyInfo;
    private boolean mUseProxy;
    private IXmppFacade.Stub mBind;

    /**
     * Constructor.
     */
    public BeemService() {
    }

    /**
     * Initialise la configuration de la connexion.
     */
    private void initConnectionConfig() {
	java.security.Security.addProvider(new com.sun.security.sasl.Provider());
	mUseProxy = mSettings.getBoolean("settings_key_proxy_use", false);
	if (mUseProxy) {
	    String stype = mSettings.getString("settings_key_proxy_type", "HTTP");
	    String phost = mSettings.getString("settings_key_proxy_server", "");
	    String puser = mSettings.getString("settings_key_proxy_username", "");
	    String ppass = mSettings.getString("settings_key_proxy_password", "");
	    int pport = Integer.parseInt(mSettings.getString("settings_key_proxy_port", "1080"));
	    ProxyInfo.ProxyType type = ProxyType.valueOf(stype);
	    mProxyInfo = new ProxyInfo(type, phost, pport, puser, ppass);
	    if (mService != null)
		mConnectionConfiguration = new ConnectionConfiguration(mHost, mPort, mService, mProxyInfo);
	    else
		mConnectionConfiguration = new ConnectionConfiguration(mHost, mPort, mProxyInfo);
	} else {
	    if (mService != null)
		mConnectionConfiguration = new ConnectionConfiguration(mHost, mPort, mService);
	    else
		mConnectionConfiguration = new ConnectionConfiguration(mHost, mPort);
	}
	if (mSettings.getBoolean("settings_key_xmpp_tls_use", false)
	    || mSettings.getBoolean("settings_key_gmail", false)) {
	    mConnectionConfiguration.setSecurityMode(SecurityMode.required);
	}
	mConnectionConfiguration.setDebuggerEnabled(false);
	mConnectionConfiguration.setSendPresence(true);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IBinder onBind(Intent intent) {
	Log.d(TAG, "ONBIND()");
	return mBind;
    }

    /* (non-Javadoc)
     * @see android.app.Service#onUnbind(android.content.Intent)
     */
    @Override
    public boolean onUnbind(Intent intent) {
	Log.d(TAG, "ONUNBIND()");
	if (!mConnection.getAdaptee().isConnected()) {
	    Log.d(TAG, "DESTROYED");
	    this.stopSelf();
	}
	return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onCreate() {

	super.onCreate();
	mSettings = PreferenceManager.getDefaultSharedPreferences(this);
	String tmpJid = mSettings.getString("settings_key_account_username", "");
	mLogin = StringUtils.parseName(tmpJid);
	mPassword = mSettings.getString("settings_key_account_password", "");
	mPort = DEFAULT_XMPP_PORT;
	mService = StringUtils.parseServer(tmpJid);
	mHost = StringUtils.parseServer(tmpJid);

	if (mSettings.getBoolean("settings_key_specific_server", false)) {
	    mHost = mSettings.getString("settings_key_xmpp_server", "");
	    mPort = Integer.parseInt(mSettings.getString("settings_key_xmpp_port", "5222"));
	}
	if (mSettings.getBoolean("settings_key_gmail", false)) {
	    mHost = "talk.google.com";
	    mLogin = tmpJid;
	}

	initConnectionConfig();
	configure(ProviderManager.getInstance());

	mNotificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
	mConnection = new XmppConnectionAdapter(mConnectionConfiguration, mLogin, mPassword, this);

	Roster.setDefaultSubscriptionMode(SubscriptionMode.manual);
	mJingle = new JingleService(mConnection.getAdaptee());
	mBind = new XmppFacade(mConnection, this, mJingle);
	Log.d(TAG, "ONCREATE");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onDestroy() {
	super.onDestroy();
	mNotificationManager.cancelAll();
	if (mConnection.isAuthentificated())
	    mConnection.disconnect();
	Log.d(TAG, "ONDESTROY");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onStart(Intent intent, int startId) {
	super.onStart(intent, startId);
	Log.d(TAG, "onStart");
	try {
	    mConnection.connectAsync();
	} catch (RemoteException e) {
	    e.printStackTrace();
	}
    }

    /**
     * Show a notification.
     * @param id the id of the notification.
     * @param notif the notification to show
     */
    public void sendNotification(int id, Notification notif) {
	mNotificationManager.notify(id, notif);
    }

    /**
     * Delete a notification.
     * @param id the id of the notification
     */
    public void deleteNotification(int id) {
	mNotificationManager.cancel(id);
    }

    /**
     * Reset the status to online after a deconnection.
     */
    public void resetStatus() {
	Editor edit = mSettings.edit();
	edit.putInt("PreferenceStatus", 1);
	edit.commit();
    }

    /**
     * Initialize Jingle from an XmppConnectionAdapter.
     * @param adaptee XmppConnection used for jingle.
     */
    public void initJingle(XMPPConnection adaptee) {
	mJingle.initWhenConntected(adaptee);
    }

    /**
     * Return a bind to an XmppFacade instance.
     * @return IXmppFacade a bind to an XmppFacade instance
     */
    public IXmppFacade getBind() {
	return mBind;
    }

    /**
     * Get the preference of the service.
     * @return the preference
     */
    public SharedPreferences getServicePreference() {
	return mSettings;
    }

    /**
     * A sort of patch from this thread: http://www.igniterealtime.org/community/thread/31118. Avoid ClassCastException
     * by bypassing the classloading shit of Smack.
     * @param pm The ProviderManager.
     */
    private void configure(ProviderManager pm) {
	// Private Data Storage
	pm.addIQProvider("query", "jabber:iq:private", new PrivateDataManager.PrivateDataIQProvider());
	// Time
	try {
	    pm.addIQProvider("query", "jabber:iq:time", Class.forName("org.jivesoftware.smackx.packet.Time"));
	} catch (ClassNotFoundException e) {
	    Log.w("TestClient", "Can't load class for org.jivesoftware.smackx.packet.Time");
	}
	// Roster Exchange
	pm.addExtensionProvider("x", "jabber:x:roster", new RosterExchangeProvider());
	// Message Events
	pm.addExtensionProvider("x", "jabber:x:event", new MessageEventProvider());
	// Chat State
	pm.addExtensionProvider("active", "http://jabber.org/protocol/chatstates", new ChatStateExtension.Provider());
	pm
	    .addExtensionProvider("composing", "http://jabber.org/protocol/chatstates",
		new ChatStateExtension.Provider());
	pm.addExtensionProvider("paused", "http://jabber.org/protocol/chatstates", new ChatStateExtension.Provider());
	pm.addExtensionProvider("inactive", "http://jabber.org/protocol/chatstates", new ChatStateExtension.Provider());
	pm.addExtensionProvider("gone", "http://jabber.org/protocol/chatstates", new ChatStateExtension.Provider());
	// XHTML
	pm.addExtensionProvider("html", "http://jabber.org/protocol/xhtml-im", new XHTMLExtensionProvider());
	// Group Chat Invitations
	pm.addExtensionProvider("x", "jabber:x:conference", new GroupChatInvitation.Provider());
	// Service Discovery # Items
	pm.addIQProvider("query", "http://jabber.org/protocol/disco#items", new DiscoverItemsProvider());
	// Service Discovery # Info
	pm.addIQProvider("query", "http://jabber.org/protocol/disco#info", new DiscoverInfoProvider());
	// Data Forms
	pm.addExtensionProvider("x", "jabber:x:data", new DataFormProvider());
	// MUC User
	pm.addExtensionProvider("x", "http://jabber.org/protocol/muc#user", new MUCUserProvider());
	// MUC Admin
	pm.addIQProvider("query", "http://jabber.org/protocol/muc#admin", new MUCAdminProvider());
	// MUC Owner
	pm.addIQProvider("query", "http://jabber.org/protocol/muc#owner", new MUCOwnerProvider());
	// Delayed Delivery
	pm.addExtensionProvider("x", "jabber:x:delay", new DelayInformationProvider());
	// Version
	try {
	    pm.addIQProvider("query", "jabber:iq:version", Class.forName("org.jivesoftware.smackx.packet.Version"));
	} catch (ClassNotFoundException e) {
	    // Not sure what's happening here.
	    Log.w("TestClient", "Can't load class for org.jivesoftware.smackx.packet.Version");
	}
	// VCard
	pm.addIQProvider("vCard", "vcard-temp", new VCardProvider());
	// Offline Message Requests
	pm.addIQProvider("offline", "http://jabber.org/protocol/offline", new OfflineMessageRequest.Provider());
	// Offline Message Indicator
	pm.addExtensionProvider("offline", "http://jabber.org/protocol/offline", new OfflineMessageInfo.Provider());
	// Last Activity
	pm.addIQProvider("query", "jabber:iq:last", new LastActivity.Provider());
	// User Search
	pm.addIQProvider("query", "jabber:iq:search", new UserSearch.Provider());
	// SharedGroupsInfo
	pm.addIQProvider("sharedgroup", "http://www.jivesoftware.org/protocol/sharedgroup",
	    new SharedGroupsInfo.Provider());
	// JEP-33: Extended Stanza Addressing
	pm.addExtensionProvider("addresses", "http://jabber.org/protocol/address", new MultipleAddressesProvider());
	// FileTransfer
	pm.addIQProvider("si", "http://jabber.org/protocol/si", new StreamInitiationProvider());
	pm.addIQProvider("query", "http://jabber.org/protocol/bytestreams", new BytestreamsProvider());
	pm.addIQProvider("open", "http://jabber.org/protocol/ibb", new IBBProviders.Open());
	pm.addIQProvider("close", "http://jabber.org/protocol/ibb", new IBBProviders.Close());
	pm.addExtensionProvider("data", "http://jabber.org/protocol/ibb", new IBBProviders.Data());
	// Privacy
	pm.addIQProvider("query", "jabber:iq:privacy", new PrivacyProvider());
	pm.addIQProvider("command", COMMAND_NAMESPACE, new AdHocCommandDataProvider());
	pm.addExtensionProvider("malformed-action", ",
	    new AdHocCommandDataProvider.MalformedActionError());
	pm.addExtensionProvider("bad-locale", COMMAND_NAMESPACE,
	    new AdHocCommandDataProvider.BadLocaleError());
	pm.addExtensionProvider("bad-payload", COMMAND_NAMESPACE,
	    new AdHocCommandDataProvider.BadPayloadError());
	pm.addExtensionProvider("bad-sessionid", COMMAND_NAMESPACE,
	    new AdHocCommandDataProvider.BadSessionIDError());
	pm.addExtensionProvider("session-expired", COMMAND_NAMESPACE,
	    new AdHocCommandDataProvider.SessionExpiredError());
    }
}
